        subroutine gauss_2(a,b,x,n)
!===========================================================
! Solutions to a system of linear equations A*x=b
! Method: Gauss elimination (with scaling and pivoting)
!-----------------------------------------------------------
! input ...
! a(n,n) - array of coefficients for matrix A
! b(n)   - array of the right hand coefficients b
! n      - number of equations (size of matrix A)
! output ...
! x(n)   - solutions
! coments ...
! the original arrays a(n,n) and b(n) will be destroyed
! during the calculation
!===========================================================
        implicit none
        integer n
        REAL*16 a(n,n), b(n), x(n)
        REAL*16 s(n)
        REAL*16 c, pivot, store
        integer i, j, k, l

! step 1: begin forward elimination
        do k=1, n-1

! step 2: "scaling"
! s(i) will have the largest element from row i
        do i=k,n
            s(i) = 0.0
            do j=k,n
                s(i) = max(s(i),abs(a(i,j)))
            end do
        end do

! step 3: "pivoting 1"
! find a row with the largest pivoting element
        pivot = abs(a(k,k)/s(k))
        l = k
        do j=k+1,n
            if(abs(a(j,k)/s(j)) > pivot) then
                pivot = abs(a(j,k)/s(j))
                l = j
            end if
        end do

! Check if the system has a sigular matrix
        if(pivot == 0.0) then
            write(*,*) ' The matrix is sigular '
            return
        end if

! step 4: "pivoting 2" interchange rows k and l (if needed)
        if (l /= k) then
            do j=k,n
                store = a(k,j)
                a(k,j) = a(l,j)
                a(l,j) = store
            end do
            store = b(k)
            b(k) = b(l)
            b(l) = store
        end if

! step 5: the elimination (after scaling and pivoting)
        do i=k+1,n
            c=a(i,k)/a(k,k)
            a(i,k) = 0.0
            b(i)=b(i)- c*b(k)
            do j=k+1,n
                a(i,j) = a(i,j)-c*a(k,j)
            end do
        end do
        end do

! step 6: back substiturion
        x(n) = b(n)/a(n,n)
        do i=n-1,1,-1
            c=0.0
            do j=i+1,n
                c= c + a(i,j)*x(j)
            end do
            x(i) = (b(i)- c)/a(i,i)
        end do

      end subroutine gauss_2
